#include <cstdlib>
#include "ringBuffer.h"
#include "CCarLifeLog.h"
// all the logging messages in this file are replaced to DLT by ADIT
LOG_IMPORT_CONTEXT(bdcl_core) //added by ADIT
ringBuffer::ringBuffer()
{

    bufferHead = NULL;
    totalBufferSize = 0;
    readIndex = 0;
    virtualTotalBufSize = 0;
    writeIndex = 0;
    availableSize = 0;
    running = true;

}

ringBuffer::~ringBuffer()
{
    delete[] bufferHead;
}

bool ringBuffer::allocateBuffer(i32 allocateSize)
{

    if (allocateSize > 0)
    {
        bufferHead = new u8[allocateSize];
        if (bufferHead != NULL)
        {
            totalBufferSize = virtualTotalBufSize = availableSize = allocateSize;
            readIndex = writeIndex = 0;
            return true;
        }
    }
    return false;
}

u32 ringBuffer::getFillLevel()
{
    return readIndex <= writeIndex ? writeIndex - readIndex : totalBufferSize - readIndex + writeIndex;
}


bool ringBuffer::readFromRingBuffer(u8* outData, u32 inLen)
{

    std::unique_lock < std::mutex > lck(ringBufMutex);

    if (inLen > totalBufferSize)
    {
        LOG_ERROR((bdcl_core,"%s: inLen is bigger than totalBufferSize", __FUNCTION__));
        return false;
    }

    while (running && getFillLevel() < inLen)
    {
        ringBufConVar.wait(lck);
    }

    if (!running)
    {

        LOG_INFO((bdcl_core,"%s: running is false \n",__FUNCTION__));
        return false;
    }

    if (readIndex < writeIndex)
    {
        memcpy(outData, (bufferHead + readIndex), inLen);
        readIndex += inLen;
    } else if (writeIndex < readIndex)
    {
        i32 remain = inLen;
        i32 offset = 0;
        u32 wrapSize = totalBufferSize - readIndex;
        if (inLen > wrapSize)
        {
            memcpy(outData, (bufferHead + readIndex), wrapSize);
            remain -= wrapSize;
            readIndex = 0;
            offset += wrapSize;
        }
        memcpy(outData + offset, (bufferHead + readIndex), remain);
        if ((readIndex + remain) >= totalBufferSize)
        {
            LOG_INFO((bdcl_core, "%s: reached end of read buffer and resetting the readIndex (%u) to bufferHead", __FUNCTION__, readIndex));
            readIndex = 0;
        } else
        {
            readIndex += remain;
        }
    }

    lck.unlock();
    ringBufConVar.notify_one();

    return true;
}

void ringBuffer::terminate()
{
    LOGD_DEBUG((bdcl_core,"%s: triggered",__FUNCTION__));
    ringBufMutex.lock();
    running = false;
    ringBufMutex.unlock();
    ringBufConVar.notify_all();
}

bool ringBuffer::writeToRingBuffer(u8* inBuffer, const u32 inDataLen)
{
    std::unique_lock < std::mutex > lck(ringBufMutex);

    if (inDataLen > totalBufferSize)
    {
        LOG_ERROR((bdcl_core,"%s: inDataLen is bigger than totalBufferSize",__FUNCTION__));
        return false;
    }

    while (running && !getAvailableBufSize(inDataLen))
    {
        ringBufConVar.wait(lck);
    }

    if (!running)
    {
        LOG_INFO((bdcl_core,"%s: running is false",__FUNCTION__));
        return false;

    }

    if (readIndex == writeIndex)
    {
        LOGD_DEBUG((bdcl_core,"%s: readIndex and writeIndex are the same value",__FUNCTION__));
        readIndex = writeIndex = 0;
        memcpy((bufferHead + writeIndex), inBuffer, inDataLen);
        writeIndex += inDataLen;

    } else if (writeIndex > readIndex)
    {
        i32 remain = inDataLen;
        i32 offset = 0;
        u32 wrapSize = totalBufferSize - writeIndex;

        if (inDataLen > wrapSize)
        {
            memcpy((bufferHead + writeIndex), inBuffer, wrapSize);
            remain -= wrapSize;
            writeIndex = 0;
            offset += wrapSize;
        }
        memcpy((bufferHead + writeIndex), inBuffer + offset, remain);
        if((writeIndex + remain) >= totalBufferSize)
        {
            LOG_INFO((bdcl_core,"%s: reached end of write buffer and resetting the writeIndex (%u) to bufferHead", __FUNCTION__, writeIndex));
            writeIndex = 0;
        }else{
            writeIndex += remain;
        }

    } else if (readIndex > writeIndex)
    {
        memcpy((bufferHead + writeIndex), inBuffer, inDataLen);
        writeIndex += inDataLen;
    }

    i32 fill =getFillLevel();
    static i32 max = 0;
    if(fill>max)
        max=fill;

    lck.unlock();
    ringBufConVar.notify_one();

    return true;
}

bool ringBuffer::getAvailableBufSize(const i32 inDataLen)
{
    i32 available_size = 0;

    if (readIndex == writeIndex)
        available_size = totalBufferSize;
    else if (writeIndex > readIndex)
        available_size = (totalBufferSize - writeIndex) + readIndex;
    else if (readIndex > writeIndex)
        available_size = readIndex - writeIndex;

    if (available_size >= inDataLen)
    {
        return true;
    }
    return false;
}
